// Scilab ( http://www.scilab.org/ ) - This file is part of Scilab
// Copyright (C) 2012 - DIGITEO - Antoine ELIAS
//
// This file must be used under the terms of the CeCILL.
// This source file is licensed as described in the file COPYING, which
// you should have received as part of this distribution.  The terms
// are also available at
// http://www.cecill.info/licences/Licence_CeCILL_V2.1-en.txt

//2012/08/06 transform macros to string to save it

//called by save function, transform handle in tlist and save result
function [] = %_save(%__filename__, varargin)

    function result = isList(var)

        //15 : list
        //16 : tlist
        //17 : mlist
        if or(type(var) == [15, 16, 17]) then
            result  = %t;
        else
            result = %f;
        end
    endfunction

    function result = isMacro(var)
        //11 : sci_u_function
        if or(type(var) == [11]) then
            result  = %t;
        else
            result = %f;
        end
    endfunction

    function result = isCompiledMacro(var)
        //11 : sci_c_function
        if or(type(var) == [13]) then
            result  = %t;
        else
            result = %f;
        end
    endfunction

    function result = inspectList(l)
        if typeof(l)=="list" then
            result = list();
            for %__i__ = definedfields(l)
                if typeof(l(%__i__)) == "handle" then
                    if ~is_handle_valid(l(%__i__)) then
                        oldMode = warning("query");
                        warning("on");
                        warning(msprintf(gettext("%s: handle no more valid ignored.\n"), "save"));
                        warning(oldMode);
                        result(%__i__) = [];
                    else
                        result(%__i__) = extractMatrixHandle(l(%__i__));
                    end
                elseif isMacro(l(%__i__)) | isCompiledMacro(l(%__i__)) then
                    //build an arbitrary name to the macro
                    result(%__i__) = extractMacro(l(%__i__), "function");
                elseif isList(l(%__i__)) then
                    result(%__i__) = inspectList(l(%__i__));
                elseif type(l(%__i__)) == 14 then //library, must not be save
                    result(%__i__) = [];
                else
                    result(%__i__) = l(%__i__);
                end
            end
        else
            fieldNb = size(getfield(1, l), "*");
            for kField = 2:fieldNb // Do not inspect first field (field names)
                fieldValue = getfield(kField, l);
                if typeof(fieldValue) == "handle" then
                    fieldValue = extractMatrixHandle(fieldValue);
                elseif isMacro(fieldValue) | isCompiledMacro(fieldValue) then
                    //build an arbitrary name to the macro
                    fieldValue = extractMacro(fieldValue, "function");
                elseif isList(fieldValue) then
                    fieldValue = inspectList(fieldValue);
                end
                setfield(kField, fieldValue, l);
            end
            result = l;
        end
    endfunction

    function matrixHandle = extractMatrixHandle(h)

        if typeof(h) <> "handle" then
            matrixHandle = [];
            return;
        end

        matrixHandle = tlist(["ScilabMatrixHandle", "dims", "values"]);
        matrixHandle.dims = size(h);
        matrixHandle.values = list();
        for %__i__ = 1:size(h, "*")
            matrixHandle.values($+1) = extractSingleHandle(h(%__i__));
            if or(fieldnames(matrixHandle.values($))=="user_data") then
                if isList(matrixHandle.values($).user_data) then
                    matrixHandle.values($).user_data = inspectList(matrixHandle.values($).user_data)
                elseif typeof(matrixHandle.values($).user_data) == "handle" then
                    matrixHandle.values($).user_data = extractMatrixHandle(matrixHandle.values($).user_data)
                end
            end
        end
    endfunction


    function item = extractSingleHandle(h)

        select h.type
        case "Figure"
            item = extractFigure(h);
        case "Axes"
            item = extractAxes(h);
        case "Polyline"
            item = extractPolyline(h);
        case "Plot3d"
            item = extractPlot3d(h);
        case "Fac3d"
            item = extractFac3d(h);
        case "Compound"
            item = extractCompound(h);
        case "Rectangle"
            item = extractRectangle(h);
        case "Arc"
            item = extractArc(h);
        case "Champ"
            item = extractChamp(h);
        case "Segs"
            item = extractSegs(h);
        case "Grayplot"
            item = extractGrayplot(h);
        case "Matplot"
            item = extractMatplot(h);
        case "Fec"
            item = extractFec(h);
        case "Legend"
            item = extractLegend(h);
        case "Text"
            item = extractText(h);
        case "Axis"
            item = extractAxis(h);
        case "uimenu"
            item = extractuimenu(h);
        case "uicontextmenu"
            item = extractuicontextmenu(h);
        case "uicontrol"
            item = extractuicontrol(h);
        case "Datatip"
            item = extractDatatip(h);
        case "Light"
            item = extractLight(h);
        else
            error("handle of type " + h.type + " unhandled");
            item = [];
        end
    endfunction

    //
    // FIGURE
    //
    function returnedFigure = extractFigure(h)
        returnedFigure = tlist([
        "ScilabSingleHandle", ...
        "type", ...
        "visible", ...
        "figure_position", ...
        "figure_size", ...
        "axes_size", ...
        "viewport", ...
        "info_message", ...
        "tag", ...
        "auto_resize", ...
        "figure_name", ...
        "figure_id", ...
        "color_map", ...
        "pixel_drawing_mode", ...
        "anti_aliasing", ...
        "immediate_drawing", ...
        "background", ...
        "rotation_style", ...
        "event_handler", ...
        "event_handler_enable", ...
        "resizefcn", ...
        "closerequestfcn", ...
        "resize", ...
        "toolbar", ...
        "toolbar_visible", ...
        "menubar", ...
        "menubar_visible", ...
        "infobar_visible", ...
        "dockable", ...
        "layout", ...
        "layout_options", ...
        "default_axes", ...
        "icon", ...
        "children", ...
        "user_data"]);

        fields = fieldnames(returnedFigure);

        for %__i__ = 1:size(fields, "*")
            if fields(%__i__) == "children" then
                returnedFigure(fields(%__i__)) = extractMatrixHandle(h(fields(%__i__)));
            else
                returnedFigure(fields(%__i__)) = h(fields(%__i__));
            end
        end
    endfunction

    //
    // LABEL
    //
    function returnedLabel = extractLabel(h)
        returnedLabel = tlist([
        "ScilabSingleHandle", ...
        "type", ...
        "visible", ...
        "text", ...
        "font_foreground", ...
        "foreground", ...
        "background", ...
        "fill_mode", ...
        "font_style", ...
        "font_size", ...
        "fractional_font", ...
        "font_angle", ...
        "auto_rotation", ...
        "position", ...
        "auto_position"]);

        fields = fieldnames(returnedLabel);

        for %__i__ = 1:size(fields, "*")
            returnedLabel(fields(%__i__)) = h(fields(%__i__));
        end
    endfunction

    //
    // TICKS
    //
    function returnedTicks = extractTicks(ticks)
        returnedTicks = tlist([
        "Ticks", ...
        "locations", ...
        "labels"]);

        fields = fieldnames(returnedTicks);

        for %__i__ = 1:size(fields, "*")
            returnedTicks(fields(%__i__)) = ticks(fields(%__i__));
        end
    endfunction

    //
    // AXES
    //
    function returnedAxes = extractAxes(h)
        returnedAxes = tlist([
        "ScilabSingleHandle", ...
        "type", ...
        "visible", ...
        "axes_visible", ...
        "axes_reverse", ...
        "grid", ...
        "grid_position", ...
        "grid_thickness", ...
        "grid_style", ...
        "x_location", ...
        "y_location", ...
        "view", ...
        "title", ...
        "x_label", ...
        "y_label", ...
        "z_label", ...
        "auto_ticks", ...
        "x_ticks", ...
        "y_ticks", ...
        "z_ticks", ...
        "ticks_format", ...
        "ticks_st", ...
        "box", ...
        "filled", ...
        "sub_tics", ...
        "font_style", ...
        "font_size", ...
        "font_color", ...
        "fractional_font", ...
        "isoview", ...
        "cube_scaling", ...
        "rotation_angles", ...
        "log_flags", ...
        "tight_limits", ...
        "data_bounds", ...
        "zoom_box", ...
        "margins", ...
        "auto_margins", ...
        "axes_bounds", ...
        "auto_clear", ...
        "auto_scale", ...
        "hidden_axis_color", ...
        "arc_drawing_method", ...
        "hiddencolor", ...
        "line_mode", ...
        "line_style", ...
        "thickness", ...
        "mark_mode", ...
        "mark_style", ...
        "mark_size", ...
        "mark_size_unit", ...
        "mark_foreground", ...
        "mark_background", ...
        "foreground", ...
        "background", ...
        "clip_state", ...
        "clip_box", ...
        "children", ...
        "user_data"]);

        fields = fieldnames(returnedAxes);

        for %__i__ = 1:size(fields, "*")
            if or(fields(%__i__) == ["title","x_label","y_label","z_label"]) then
                returnedAxes(fields(%__i__)) = extractLabel(h(fields(%__i__)));
            elseif or(fields(%__i__) == ["x_ticks", "y_ticks", "z_ticks"]) then
                returnedAxes(fields(%__i__)) = extractTicks(h(fields(%__i__)));
            elseif fields(%__i__) == "children" then
                returnedAxes(fields(%__i__)) = extractMatrixHandle(h(fields(%__i__)));
            else
                returnedAxes(fields(%__i__)) = h(fields(%__i__));
            end
        end
    endfunction

    //
    // POLYLINE
    //
    function returnedPolyline = extractPolyline(h)
        returnedPolyline = tlist([
        "ScilabSingleHandle", ...
        "type", ...
        "visible", ...
        "datatips", ...
        "display_function", ...
        "display_function_data", ...
        "data", ...
        "closed", ...
        "line_mode", ...
        "fill_mode", ...
        "line_style", ...
        "thickness", ...
        "arrow_size_factor", ...
        "polyline_style", ...
        "interp_color_vector", ...
        "interp_color_mode", ...
        "mark_mode", ...
        "mark_style", ...
        "mark_size", ...
        "mark_size_unit", ...
        "foreground", ...
        "background", ...
        "mark_foreground", ...
        "mark_background", ...
        "mark_offset", ...
        "mark_stride", ...
        "x_shift", ...
        "y_shift", ...
        "z_shift", ...
        "bar_width", ...
        "clip_state", ...
        "clip_box", ...
        "user_data"]);

        fields = fieldnames(returnedPolyline);

        for %__i__ = 1:size(fields, "*")
            if or(fields(%__i__) == ["children", "datatips"]) then
                returnedPolyline(fields(%__i__)) = extractMatrixHandle(h(fields(%__i__)));
            else
                returnedPolyline(fields(%__i__)) = h(fields(%__i__));
            end
        end
    endfunction

    //
    // PLOT3D
    //
    function returnedPlot3d = extractPlot3d(h)
        returnedPlot3d = extractSurface(h);
    endfunction

    //
    // FAC3D
    //
    function returnedFac3d = extractFac3d(h)
        returnedFac3d = extractSurface(h);
    endfunction

    //
    // SURFACE
    //
    function returnedSurface = extractSurface(h)
        returnedSurface = tlist([
        "ScilabSingleHandle", ...
        "type", ...
        "visible", ...
        "surface_mode", ...
        "foreground", ...
        "thickness", ...
        "mark_mode", ...
        "mark_style", ...
        "mark_size", ...
        "mark_size_unit", ...
        "mark_foreground", ...
        "mark_background", ...
        "color_mode", ...
        "color_flag", ...
        "data", ...
        "color_flag", ...
        "ambient_color", ...
        "diffuse_color", ...
        "specular_color", ...
        "use_color_material", ...
        "material_shininess", ...
        "hiddencolor", ...
        "clip_state", ...
        "clip_box", ...
        "user_data", ...
        "cdata_mapping"]);

        fields = fieldnames(returnedSurface);

        for %__i__ = 1:size(fields, "*")
            if fields(%__i__)=="cdata_mapping" then
                if h.type=="Fac3d" then
                    returnedSurface(fields(%__i__)) = h(fields(%__i__));
                end
            else
                returnedSurface(fields(%__i__)) = h(fields(%__i__));
            end
        end
    endfunction

    //
    // COMPOUND
    //
    function returnedCompound = extractCompound(h)
        returnedCompound = tlist([
        "ScilabSingleHandle", ...
        "type", ...
        "visible", ...
        "children", ...
        "user_data"]);

        fields = fieldnames(returnedCompound);

        for %__i__ = 1:size(fields, "*")
            if fields(%__i__) == "children" then
                returnedCompound(fields(%__i__)) = extractMatrixHandle(h(fields(%__i__)));
            else
                returnedCompound(fields(%__i__)) = h(fields(%__i__));
            end
        end
    endfunction

    //
    // RECTANGLE
    //
    function returnedRectangle = extractRectangle(h)
        returnedRectangle = tlist([
        "ScilabSingleHandle", ...
        "type", ...
        "visible", ...
        "thickness", ...
        "mark_mode", ...
        "mark_style", ...
        "mark_size", ...
        "mark_size_unit", ...
        "mark_foreground", ...
        "mark_background", ...
        "line_mode", ...
        "line_style", ...
        "fill_mode", ...
        "foreground", ...
        "background", ...
        "data", ...
        "clip_state", ...
        "clip_box", ...
        "user_data"]);

        fields = fieldnames(returnedRectangle);

        for %__i__ = 1:size(fields, "*")
            returnedRectangle(fields(%__i__)) = h(fields(%__i__));
        end
    endfunction

    //
    // ARC
    //
    function returnedArc = extractArc(h)
        returnedArc = tlist([
        "ScilabSingleHandle", ...
        "type", ...
        "visible", ...
        "thickness", ...
        "line_style", ...
        "line_mode", ...
        "fill_mode", ...
        "foreground", ...
        "background", ...
        "data", ...
        "arc_drawing_method", ...
        "clip_state", ...
        "clip_box", ...
        "user_data"]);

        fields = fieldnames(returnedArc);

        for %__i__ = 1:size(fields, "*")
            returnedArc(fields(%__i__)) = h(fields(%__i__));
        end
    endfunction

    //
    // CHAMP
    //
    function returnedChamp = extractChamp(h)
        returnedChamp = tlist([
        "ScilabSingleHandle", ...
        "type", ...
        "visible", ...
        "data", ...
        "line_style", ...
        "thickness", ...
        "colored", ...
        "arrow_size", ...
        "clip_state", ...
        "clip_box", ...
        "user_data"]);

        fields = fieldnames(returnedChamp);

        for %__i__ = 1:size(fields, "*")
            returnedChamp(fields(%__i__)) = h(fields(%__i__));
        end
    endfunction

    //
    // SEG
    //
    function returnedSeg = extractSegs(h)
        returnedSeg = tlist([
        "ScilabSingleHandle", ...
        "type", ...
        "visible", ...
        "data", ...
        "line_mode", ...
        "line_style", ...
        "thickness", ...
        "segs_color", ...
        "mark_mode", ...
        "mark_style", ...
        "mark_size", ...
        "mark_size_unit", ...
        "mark_foreground", ...
        "mark_background", ...
        "clip_state", ...
        "clip_box", ...
        "user_data"]);

        fields = fieldnames(returnedSeg);

        for %__i__ = 1:size(fields, "*")
            returnedSeg(fields(%__i__)) = h(fields(%__i__));
        end
    endfunction

    //
    // GRAYPLOT
    //
    function returnedGrayplot = extractGrayplot(h)
        returnedGrayplot = tlist([
        "ScilabSingleHandle", ...
        "type", ...
        "visible", ...
        "data", ...
        "data_mapping", ...
        "clip_state", ...
        "clip_box", ...
        "user_data"]);

        fields = fieldnames(returnedGrayplot);

        for %__i__ = 1:size(fields, "*")
            returnedGrayplot(fields(%__i__)) = h(fields(%__i__));
        end
    endfunction

    //
    // MATPLOT
    //
    function returnedMatplot = extractMatplot(h)
        returnedMatplot = tlist([
        "ScilabSingleHandle", ...
        "type", ...
        "visible", ...
        "data", ...
        "rect", ...
        "image_type", ...
        "clip_state", ...
        "clip_box", ...
        "user_data"]);

        fields = fieldnames(returnedMatplot);

        for %__i__ = 1:size(fields, "*")
            returnedMatplot(fields(%__i__)) = h(fields(%__i__));
        end
    endfunction

    //
    // FEC
    //
    function returnedFec = extractFec(h)
        returnedFec = tlist([
        "ScilabSingleHandle", ...
        "type", ...
        "visible", ...
        "data", ...
        "triangles", ...
        "z_bounds", ...
        "color_range", ...
        "outside_colors", ...
        "line_mode", ...
        "foreground", ...
        "clip_state", ...
        "clip_box", ...
        "user_data"]);

        fields = fieldnames(returnedFec);

        for %__i__ = 1:size(fields, "*")
            returnedFec(fields(%__i__)) = h(fields(%__i__));
        end
    endfunction

    //
    // LEGEND
    //
    function returnedLegend = extractLegend(h)
        returnedLegend = tlist([
        "ScilabSingleHandle", ...
        "type", ...
        "visible", ...
        "text", ...
        "font_style", ...
        "font_size", ...
        "font_color", ...
        "fractional_font", ...
        "links", ...
        "legend_location", ...
        "position", ...
        "line_width", ...
        "line_mode", ...
        "thickness", ...
        "foreground", ...
        "fill_mode", ...
        "background", ...
        "marks_count", ...
        "clip_state", ...
        "clip_box", ...
        "user_data", ..
        "paths"]);

        fields = fieldnames(returnedLegend);

        for %__i__ = 1:size(fields, "*")
            if fields(%__i__) == "links" then
                returnedLegend(fields(%__i__)) = extractMatrixHandle(h(fields(%__i__)));
            elseif fields(%__i__) == "paths" then
                p = list();
                for kl=1:size(h.links,"*");
                    p($+1) = get_entity_path(h.links(kl));
                end
                returnedLegend(fields(%__i__)) = p;
            else
                returnedLegend(fields(%__i__)) = h(fields(%__i__));
            end
        end
    endfunction

    //
    // TEXT
    //
    function returnedText = extractText(h)
        returnedText = tlist([
        "ScilabSingleHandle", ...
        "type", ...
        "visible", ...
        "text", ...
        "data", ...
        "text_box", ...
        "text_box_mode", ...
        "foreground", ...
        "font_style", ...
        "font_size", ...
        "font_angle", ...
        "box", ...
        "line_mode", ...
        "fill_mode", ...
        "font_foreground", ...
        "background", ...
        "alignment", ...
        "fractional_font", ...
        "clip_state", ...
        "clip_box", ...
        "user_data"]);

        fields = fieldnames(returnedText);

        for %__i__ = 1:size(fields, "*")
            returnedText(fields(%__i__)) = h(fields(%__i__));
        end
    endfunction

    //
    // DATATIP
    //
    function returnedDatatip = extractDatatip(h)
        returnedDatatip = tlist([
        "ScilabSingleHandle", ...
        "type", ...
        "visible", ...
        "data", ...
        "box_mode", ...
        "label_mode", ...
        "orientation", ...
        "z_component", ...
        "auto_orientation", ...
        "interp_mode", ...
        "display_function", ...
        "font_foreground", ...
        "foreground", ...
        "background", ...
        "mark_mode", ...
        "mark_style", ...
        "mark_size_unit", ...
        "mark_size", ...
        "mark_foreground", ...
        "mark_background", ...
        "user_data", ...
        ]);

        fields = fieldnames(returnedDatatip);

        for %__i__ = 1:size(fields, "*")
            returnedDatatip(fields(%__i__)) = h(fields(%__i__));
        end
    endfunction

    //
    // AXIS
    //
    function returnedAxis = extractAxis(h)
        returnedAxis = tlist([
        "ScilabSingleHandle", ...
        "type", ...
        "visible", ...
        "tics_direction", ...
        "xtics_coord", ...
        "ytics_coord", ...
        "tics_color", ...
        "tics_segment", ...
        "tics_style", ...
        "sub_tics", ...
        "tics_labels", ...
        "labels_font_size", ...
        "labels_font_color", ...
        "fractional_font", ...
        "clip_state", ...
        "clip_box", ...
        "user_data"]);

        fields = fieldnames(returnedAxis);

        for %__i__ = 1:size(fields, "*")
            returnedAxis(fields(%__i__)) = h(fields(%__i__));
        end
    endfunction

    //
    // uimenu
    //
    function returneduimenu = extractuimenu(h)
        returneduimenu = tlist([
        "ScilabSingleHandle", ...
        "type", ...
        "enable", ...
        "foregroundcolor", ...
        "label", ...
        "visible", ...
        "callback", ...
        "callback_type", ...
        "tag", ...
        "checked", ...
        "children", ...
        "icon", ...
        "user_data"]);

        fields = fieldnames(returneduimenu);

        for %__i__ = 1:size(fields, "*")
            if fields(%__i__) == "children" then
                returneduimenu(fields(%__i__)) = extractMatrixHandle(h(fields(%__i__)));
            else
                returneduimenu(fields(%__i__)) = h(fields(%__i__));
            end
        end
    endfunction

    //
    // UICONTEXTMENU
    //
    function returneduicontextmenu = extractuicontextmenu(h)
        returneduicontextmenu = tlist([
        "ScilabSingleHandle", ...
        "type", ...
        "children"]);

        fields = fieldnames(returneduicontextmenu);

        for %__i__ = 1:size(fields, "*")
            if fields(%__i__) == "children" then
                returneduicontextmenu(fields(%__i__)) = extractMatrixHandle(h(fields(%__i__)));
            else
                returneduicontextmenu(fields(%__i__)) = h(fields(%__i__));
            end
        end
    endfunction

    //
    // uicontrol
    //
    function returneduicontrol = extractuicontrol(h)
        returneduicontrol = tlist([
        "ScilabSingleHandle", ...
        "type", ...
        "style", ...
        "backgroundcolor", ...
        "enable", ...
        "fontangle", ...
        "fontname", ...
        "fontsize", ...
        "fontunits", ...
        "fontweight", ...
        "foregroundcolor", ...
        "horizontalalignment", ...
        "listboxtop", ...
        "max", ...
        "min", ...
        "position", ...
        "relief", ...
        "sliderstep", ...
        "string", ...
        "tooltipstring", ...
        "units", ...
        "value", ...
        "verticalalignment", ...
        "visible", ...
        "callback", ...
        "callback_type", ...
        "layout", ...
        "layout_options", ...
        "constraints", ...
        "border", ...
        "margins", ...
        "groupname", ...
        "scrollable", ...
        "icon", ...
        "user_data", ...
        "tag", ...
        "children"]);

        fields = fieldnames(returneduicontrol);

        for %__i__ = 1:size(fields, "*")
            if fields(%__i__) == "children" then
                returneduicontrol(fields(%__i__)) = extractMatrixHandle(h(fields(%__i__)));
            else
                returneduicontrol(fields(%__i__)) = h(fields(%__i__));
            end
        end
    endfunction

    //
    // LIGHT
    //
    function returnedLight = extractLight(h)
        returnedLight = tlist([
        "ScilabSingleHandle", ...
        "type", ...
        "visible", ...
        "light_type", ...
        "position", ...
        "direction", ...
        "ambient_color", ...
        "diffuse_color", ...
        "specular_color"]);

        fields = fieldnames(returnedLight);

        for %__i__ = 1:size(fields, "*")
            returnedLight(fields(%__i__)) = h(fields(%__i__));
        end
    endfunction

    // Utility function for legends, copy/paste from %h_load
    function p=get_entity_path(e)
        // given a handle e on an entity this function returns its path relative
        // to its parent axes.
        // the path is a vector of child index.
        p=[];
        parent=e.parent;

        while  %t
            pos=find(parent.children==e,1)
            if pos==[] then
                error(msprintf(_("%s : Invalid entity %s\n"),"save","Legend"))
            end
            p=[pos p]
            if parent.type=="Axes" then
                break
            end
            e=parent
            parent=e.parent;
        end
    endfunction

    function macro = extractMacro(macroPtr, macroName)
        macroSt = fun2string(macroPtr, macroName);
        oldMode = warning("query");
        warning("off");
        macro = tlist("ScilabMacro", isCompiledMacro(macroPtr), macroSt);
        warning(oldMode);
    endfunction

    //main

    //save environment
    if size(varargin) == 0 then
        %__excludeList__ = [
        "%_save"
        "isList"
        "isMacro"
        "isCompiledMacro"
        "inspectList"
        "extractMatrixHandle"
        "extractSingleHandle"
        "extractFigure"
        "extractLabel"
        "extractTicks"
        "extractAxes"
        "extractPolyline"
        "extractPlot3d"
        "extractFac3d"
        "extractSurface"
        "extractCompound"
        "extractRectangle"
        "extractArc"
        "extractChamp"
        "extractSegs"
        "extractGrayplot"
        "extractMatplot"
        "extractFec"
        "extractLegend"
        "extractText"
        "extractAxis"
        "extractuimenu"
        "extractuicontextmenu"
        "extractuicontrol"
        "get_entity_path"
        "extractMacro"
        "%__excludeList__"
        "%__filename__"
        "varargin"
        "%__varList__"];

        //get all user variables
        %__varList__ = who_user(%f);
        //remove exclude variables/functions
        %__grepResult__ = grep(%__varList__, %__excludeList__);
        %__varList__(%__grepResult__) = [];
        for %__i__ = 1:size(%__varList__, "*")
            //store them as input arguments
            varargin(%__i__) = %__varList__(%__i__);
        end
    end

    oldMode = warning("query");
    warning("off");

    if size(varargin) == 0 then
    end

    for %__i__ = size(varargin):-1:1

        if varargin(%__i__) == "-append" then
            continue;
        end

        %__temp__ = evstr(varargin(%__i__));

        if isList(%__temp__) then
            //list container
            value = inspectList(%__temp__);
            //update
            execstr(varargin(%__i__) + " = value");
        elseif typeof(%__temp__) == "handle" then
            if ~is_handle_valid(%__temp__) then // Invalid handle ignored
                warning(oldMode);
                warning(msprintf(gettext("%s: handle no more valid ignored.\n"), "save"));
                warning("off");
                varargin(%__i__) = null();
            else //convert handle to tlist
                value = extractMatrixHandle(%__temp__);
                //update
                execstr(varargin(%__i__) + " = value");
            end
        elseif isMacro(%__temp__) | isCompiledMacro(%__temp__) then
            //convert macro to tlist
            value = extractMacro(%__temp__, varargin(%__i__));
            //update
            execstr(varargin(%__i__) + " = value");
        elseif type(%__temp__) == 14 then //library, must not be save
            varargin(%__i__) = null();
        end
    end
    warning(oldMode);

    //if size(varargin) == 0, create an empty file
    result = export_to_hdf5(%__filename__, varargin(:));

endfunction
